#include <algorithm>
#include <exception>
#include <cfloat>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <map>
#include <vector>


namespace DFF {
  class Exception : public std::exception { };
}

namespace Dinamica {
  class ImageException : public DFF::Exception { };

  class BaseRasterImage;

  class LookupTable;
}


typedef unsigned short CoordinateValue;

typedef unsigned int IndexValue;


extern "C" {

bool NeighborhoodQuaternaryExpression_checkWindowAttributesDouble( double windowLines, double windowCols,
  double centerLine, double centerCol );

bool NeighborhoodQuaternaryExpression_checkWindowAttributesFloat( float windowLines, float windowCols,
  float centerLine, float centerCol );


void NeighborhoodQuaternaryExpression_translateWindowCoordinates( CoordinateValue *imageLine,
  CoordinateValue *imageColumn, CoordinateValue windowLine, CoordinateValue windowColumn,
  CoordinateValue imageLines, CoordinateValue imageColumns,
  CoordinateValue windowLines, CoordinateValue windowColumns,
  CoordinateValue centerLine, CoordinateValue centerColumn );


bool Real_areCloseDouble( double a, double b );

bool Real_areCloseFloat( float a, float b );


double Random_getUniformDouble();

float Random_getUniformFloat();


bool Real_isInvalidDouble( double value );

bool Real_isInvalidFloat( float value );


double Real_invalidDouble();

float Real_invalidFloat();


double LookupTable_getEqualLowerBoundValueDouble( const Dinamica::LookupTable* table, 
  double key, double defaultValue );

float LookupTable_getEqualLowerBoundValueFloat( const Dinamica::LookupTable* table, 
  float key, float defaultValue );


double LookupTable_getLowerBoundValueDouble( const Dinamica::LookupTable* table,
  double key, double defaultValue );

float LookupTable_getLowerBoundValueFloat( const Dinamica::LookupTable* table,
  float key, float defaultValue );


double LookupTable_getEqualUpperBoundValueDouble( const Dinamica::LookupTable* table,
  double key, double defaultValue );

float LookupTable_getEqualUpperBoundValueFloat( const Dinamica::LookupTable* table,
  float key, float defaultValue );


double LookupTable_getUpperBoundValueDouble( const Dinamica::LookupTable* table,
  double key, double defaultValue );

float LookupTable_getUpperBoundValueFloat( const Dinamica::LookupTable* table,
  float key, float defaultValue );


double LookupTable_getClosestValueDouble( const Dinamica::LookupTable* table,
  double key, double defaultValue );

float LookupTable_getClosestValueFloat( const Dinamica::LookupTable* table,
  float key, float defaultValue );


double LookupTable_getInterpoledValueDouble( const Dinamica::LookupTable* table,
  double key, double defaultValue );

float LookupTable_getInterpoledValueFloat( const Dinamica::LookupTable* table,
  float key, float defaultValue );


bool LookupTable_hasKeyDouble( const Dinamica::LookupTable* table, double key );

bool LookupTable_hasKeyFloat( const Dinamica::LookupTable* table, float key );
  
 
double LookupTable_getValueDouble( const Dinamica::LookupTable* table, 
  double key, double defaultValue );

float LookupTable_getValueFloat( const Dinamica::LookupTable* table, 
  float key, float defaultValue );


void LookupTable_setValueDouble( Dinamica::LookupTable* table, 
  double key, double value );

void LookupTable_setValueFloat( Dinamica::LookupTable* table, 
  float key, float value );


double BaseRasterImage_getValueDouble( const Dinamica::BaseRasterImage* image, 
  CoordinateValue line, CoordinateValue column );

float BaseRasterImage_getValueFloat( const Dinamica::BaseRasterImage* image, 
  CoordinateValue line, CoordinateValue column );
  
  
double BaseRasterImage_getValueDouble2( const Dinamica::BaseRasterImage* image, 
  double line, double column, double nullValue );

float BaseRasterImage_getValueFloat2( const Dinamica::BaseRasterImage* image, 
  float line, float column, float nullValue );


CoordinateValue BaseRasterImage_getLines( const Dinamica::BaseRasterImage* image );


CoordinateValue BaseRasterImage_getColumns( const Dinamica::BaseRasterImage* image );
  
  
void DFF_throwExecutionAbortedException();


}


namespace NeighborhoodQuaternaryExpression {

  template < typename ValueT >
  bool checkWindowAttributes( ValueT windowLines, ValueT windowCols,
    ValueT centerLine, ValueT centerCol );
  template <>
  bool checkWindowAttributes< double >( double windowLines, double windowCols,
    double centerLine, double centerCol ) {
    return NeighborhoodQuaternaryExpression_checkWindowAttributesDouble( windowLines, windowCols,
      centerLine, centerCol );
  }
  template <>
  bool checkWindowAttributes< float >( float windowLines, float windowCols,
    float centerLine, float centerCol ) {
    return NeighborhoodQuaternaryExpression_checkWindowAttributesFloat( windowLines, windowCols,
      centerLine, centerCol );
  }

  inline void translateWindowCoordinates( CoordinateValue& imageLine, CoordinateValue& imageColumn,
	CoordinateValue windowLine, CoordinateValue windowColumn,
	CoordinateValue imageLines, CoordinateValue imageColumns,
	CoordinateValue windowLines, CoordinateValue windowColumns,
	CoordinateValue centerLine, CoordinateValue centerColumn ) {
    NeighborhoodQuaternaryExpression_translateWindowCoordinates( &imageLine, &imageColumn,
      windowLine, windowColumn, imageLines, imageColumns, windowLines, windowColumns,
      centerLine, centerColumn );
  }
}


namespace Real {

  template < typename ValueT >
  bool areClose( ValueT a, ValueT b );
  template <>
  bool areClose< double >( double a, double b ) {
    return Real_areCloseDouble( a,  b );
  }
  template <>
  bool areClose< float >( float a, float b ) {
    return Real_areCloseFloat( a,  b );
  }

  template < typename ValueT >
  bool isInvalid( ValueT value );
  template <>
  bool isInvalid< double >( double value ) {
    return Real_isInvalidDouble( value );
  }
  template <>
  bool isInvalid< float >( float value ) {
    return Real_isInvalidFloat( value );
  }

  template < typename ValueT >
  ValueT invalid();
  template <>
  double invalid< double >() {
    return Real_invalidDouble();
  }
  template <>
  float invalid< float >() {
    return Real_invalidFloat();
  }
}


namespace Random {

  template < typename ValueT >
  ValueT getUniform();
  template <>
  double getUniform< double >() {
    return Random_getUniformDouble();
  }
  template <>
  float getUniform< float >() {
    return Random_getUniformFloat();
  }
}


namespace LookupTable {

  template < typename ValueT >
  ValueT getEqualLowerBoundValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getEqualLowerBoundValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getEqualLowerBoundValueDouble( table, key, defaultValue );
  }
  template <>
  float getEqualLowerBoundValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getEqualLowerBoundValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  ValueT getLowerBoundValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getLowerBoundValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getLowerBoundValueDouble( table, key, defaultValue );
  }
  template <>
  float getLowerBoundValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getLowerBoundValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  ValueT getEqualUpperBoundValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getEqualUpperBoundValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getEqualUpperBoundValueDouble( table, key, defaultValue );
  }
  template <>
  float getEqualUpperBoundValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getEqualUpperBoundValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  ValueT getUpperBoundValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getUpperBoundValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getUpperBoundValueDouble( table, key, defaultValue );
  }
  template <>
  float getUpperBoundValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getUpperBoundValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  ValueT getClosestValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getClosestValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getClosestValueDouble( table, key, defaultValue );
  }
  template <>
  float getClosestValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getClosestValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  ValueT getInterpoledValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getInterpoledValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getInterpoledValueDouble( table, key, defaultValue );
  }
  template <>
  float getInterpoledValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getInterpoledValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  bool hasKey( const Dinamica::LookupTable* table, ValueT key );
  template <>
  bool hasKey< double >( const Dinamica::LookupTable* table, double key ) {
    return LookupTable_hasKeyDouble( table, key );
  }
  template <>
  bool hasKey< float >( const Dinamica::LookupTable* table, float key ) {
    return LookupTable_hasKeyFloat( table, key );
  }
 
  template < typename ValueT >
  ValueT getValue( const Dinamica::LookupTable* table,
    ValueT key, ValueT defaultValue );
  template <>
  double getValue< double >( const Dinamica::LookupTable* table,
    double key, double defaultValue ) {
    return LookupTable_getValueDouble( table, key, defaultValue );
  }
  template <>
  float getValue< float >( const Dinamica::LookupTable* table,
    float key, float defaultValue ) {
    return LookupTable_getValueFloat( table, key, defaultValue );
  }

  template < typename ValueT >
  void setValue( Dinamica::LookupTable* table,
    ValueT key, ValueT value );
  template <>
  void setValue< double >( Dinamica::LookupTable* table,
    double key, double value ) {
    LookupTable_setValueDouble( table, key, value );
  }
  template <>
  void setValue< float >( Dinamica::LookupTable* table,
    float key, float value ) {
    LookupTable_setValueFloat( table, key, value );
  }
}


namespace BaseRasterImage {

  template < typename ValueT >
  ValueT getValue( const Dinamica::BaseRasterImage* image,
    CoordinateValue line, CoordinateValue column );
  template <>
  double getValue< double >( const Dinamica::BaseRasterImage* image,
    CoordinateValue line, CoordinateValue column ) {
    return BaseRasterImage_getValueDouble( image, line, column );
  }
  template <>
  float getValue< float >( const Dinamica::BaseRasterImage* image,
    CoordinateValue line, CoordinateValue column ) {
    return BaseRasterImage_getValueFloat( image, line, column );
  }

  template < typename ValueT >
  ValueT getValue2( const Dinamica::BaseRasterImage* image,
    ValueT line, ValueT column, ValueT nullValue );
  template <>
  double getValue2< double >( const Dinamica::BaseRasterImage* image,
    double line, double column, double nullValue ) {
    return BaseRasterImage_getValueDouble2( image, line, column, nullValue );
  }
  template <>
  float getValue2< float >( const Dinamica::BaseRasterImage* image,
    float line, float column, float nullValue ) {
    return BaseRasterImage_getValueFloat2( image, line, column, nullValue );
  }

  inline CoordinateValue getLines( const Dinamica::BaseRasterImage* image ) {
    return BaseRasterImage_getLines( image );
  }

  inline CoordinateValue getColumns( const Dinamica::BaseRasterImage* image ) {
    return BaseRasterImage_getColumns( image );
  }
}
